"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SignatureTransfer = void 0;
const tslib_1 = require("tslib");
const tiny_invariant_1 = tslib_1.__importDefault(require("tiny-invariant"));
const hash_1 = require("@ethersproject/hash");
const domain_1 = require("./domain");
const constants_1 = require("./constants");
const TOKEN_PERMISSIONS = [
    { name: 'token', type: 'address' },
    { name: 'amount', type: 'uint256' },
];
const PERMIT_TRANSFER_FROM_TYPES = {
    PermitTransferFrom: [
        { name: 'permitted', type: 'TokenPermissions' },
        { name: 'spender', type: 'address' },
        { name: 'nonce', type: 'uint256' },
        { name: 'deadline', type: 'uint256' },
    ],
    TokenPermissions: TOKEN_PERMISSIONS,
};
const PERMIT_BATCH_TRANSFER_FROM_TYPES = {
    PermitBatchTransferFrom: [
        { name: 'permitted', type: 'TokenPermissions[]' },
        { name: 'spender', type: 'address' },
        { name: 'nonce', type: 'uint256' },
        { name: 'deadline', type: 'uint256' },
    ],
    TokenPermissions: TOKEN_PERMISSIONS,
};
function permitTransferFromWithWitnessType(witness) {
    return Object.assign({ PermitWitnessTransferFrom: [
            { name: 'permitted', type: 'TokenPermissions' },
            { name: 'spender', type: 'address' },
            { name: 'nonce', type: 'uint256' },
            { name: 'deadline', type: 'uint256' },
            { name: 'witness', type: witness.witnessTypeName },
        ], TokenPermissions: TOKEN_PERMISSIONS }, witness.witnessType);
}
function permitBatchTransferFromWithWitnessType(witness) {
    return Object.assign({ PermitBatchWitnessTransferFrom: [
            { name: 'permitted', type: 'TokenPermissions[]' },
            { name: 'spender', type: 'address' },
            { name: 'nonce', type: 'uint256' },
            { name: 'deadline', type: 'uint256' },
            { name: 'witness', type: witness.witnessTypeName },
        ], TokenPermissions: TOKEN_PERMISSIONS }, witness.witnessType);
}
function isPermitTransferFrom(permit) {
    return !Array.isArray(permit.permitted);
}
class SignatureTransfer {
    /**
     * Cannot be constructed.
     */
    constructor() { }
    // return the data to be sent in a eth_signTypedData RPC call
    // for signing the given permit data
    static getPermitData(permit, permit2Address, chainId, witness) {
        (0, tiny_invariant_1.default)(constants_1.MaxSigDeadline.gte(permit.deadline), 'SIG_DEADLINE_OUT_OF_RANGE');
        (0, tiny_invariant_1.default)(constants_1.MaxUnorderedNonce.gte(permit.nonce), 'NONCE_OUT_OF_RANGE');
        const domain = (0, domain_1.permit2Domain)(permit2Address, chainId);
        if (isPermitTransferFrom(permit)) {
            validateTokenPermissions(permit.permitted);
            const types = witness ? permitTransferFromWithWitnessType(witness) : PERMIT_TRANSFER_FROM_TYPES;
            const values = witness ? Object.assign(permit, { witness: witness.witness }) : permit;
            return {
                domain,
                types,
                values,
            };
        }
        else {
            permit.permitted.forEach(validateTokenPermissions);
            const types = witness ? permitBatchTransferFromWithWitnessType(witness) : PERMIT_BATCH_TRANSFER_FROM_TYPES;
            const values = witness ? Object.assign(permit, { witness: witness.witness }) : permit;
            return {
                domain,
                types,
                values,
            };
        }
    }
    static hash(permit, permit2Address, chainId, witness) {
        const { domain, types, values } = SignatureTransfer.getPermitData(permit, permit2Address, chainId, witness);
        return hash_1._TypedDataEncoder.hash(domain, types, values);
    }
}
exports.SignatureTransfer = SignatureTransfer;
function validateTokenPermissions(permissions) {
    (0, tiny_invariant_1.default)(constants_1.MaxSignatureTransferAmount.gte(permissions.amount), 'AMOUNT_OUT_OF_RANGE');
}
//# sourceMappingURL=signatureTransfer.js.map